# -*- Mode: Python; coding: utf-8; indent-tabs-mode: nil; tab-width: 4 -*-
### BEGIN LICENSE
# Copyright (C) 2012 Pete Burgers <deltify81@gmail.com>
# This program is free software: you can redistribute it and/or modify it 
# under the terms of the GNU General Public License version 3, as published 
# by the Free Software Foundation.
# 
# This program is distributed in the hope that it will be useful, but 
# WITHOUT ANY WARRANTY; without even the implied warranties of 
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
# PURPOSE.  See the GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License along 
# with this program.  If not, see <http://www.gnu.org/licenses/>.
### END LICENSE

import gettext
from gettext import gettext as _
gettext.textdomain('deltify')

import logging
logger = logging.getLogger(__name__)

from gi.repository import Gtk # pylint: disable=E0611
from gi.repository import GObject

import deltify.browser as browser
from deltify_lib.Assistant import Assistant

# See deltify_lib.Assistant.py for more details about how this class works.
class DeltifyAssistant(Assistant):
    __gtype_name__ = "DeltifyAssistant"
    
    def finish_initializing(self, builder): # pylint: disable=E1002
        """Set up the deltify assistant"""
        super(DeltifyAssistant, self).finish_initializing(builder)

        self.browser_buttons = self.builder.get_object("firefox_button").get_group()
        self.profile_combobox = self.builder.get_object("profile_combobox")
        self.builder.get_object("firefox_icon").show()
        self.builder.get_object("chrome_icon").show()
        self.builder.get_object("chromium_icon").show()

        self.browser_page = self.builder.get_object("browser_page")
        self.profile_page = self.builder.get_object("profile_page")
        self.bookmark_root_page = self.builder.get_object("bookmark_root_page")
        self.summary_page = self.builder.get_object("summary_page")

        title = _("Welcome to Deltify!")
        self.setup_page(self.browser_page, title=title,
                        page_type=Gtk.AssistantPageType.INTRO)
        self.setup_page(self.profile_page, title=title, complete=False)
# The world's not ready for this yet
#        self.setup_page(self.bookmark_root_page, title=title)
        self.setup_page(self.summary_page, title=title,
                        page_type=Gtk.AssistantPageType.SUMMARY)

        self.parent_on_close = None
        self.parent_on_cancel = None

    def show(self, on_close, on_cancel=None):
        self.parent_on_close = on_close
        self.parent_on_cancel = on_cancel
        super(DeltifyAssistant, self).show()

    def on_apply(self, widget):
        print "Apply!"

    def on_prepare(self, widget, page):
        if page == self.profile_page:
            self.on_prepare_profile_page()

    def on_prepare_profile_page(self):
        browser_name = self.get_browser()
        profiles = browser.browser_from_name(browser_name).get_profiles()

        if not profiles:
            self.show_profile_error(browser_name)
            GObject.idle_add(self.previous_page)
            return

        self.profile_combobox.remove_all()
        for name, path in profiles:
            self.profile_combobox.append(id=path, text=name)
        # Select the first profile by default
        self.profile_combobox.set_active(0)

    def on_profile_combobox_changed(self, widget):
        complete = self.get_profile() is not None
        self.set_page_complete(self.profile_page, complete)

    def on_close(self, widget):
        self.destroy()
        if self.parent_on_close:
            self.parent_on_close(self.get_browser(), self.get_profile())

    def on_cancel(self, widget):
        self.destroy()
        if self.parent_on_cancel:
            self.parent_on_cancel()

    def setup_page(self, page, title, page_type=Gtk.AssistantPageType.CONTENT, complete=True):
        self.append_page(page)
        self.set_page_title(page, title)
        self.set_page_type(page, page_type)
        self.set_page_complete(page, complete)
        
    def get_browser(self):
        browser_button = [button for button in self.browser_buttons 
                          if button.get_active()][0]
        return Gtk.Buildable.get_name(browser_button).replace("_button", "")

    def get_profile(self):
        return self.profile_combobox.get_active_id()

    def show_profile_error(self, browser_name):
        browser_name = browser_name.capitalize()
        dialog = Gtk.MessageDialog(self, Gtk.DialogFlags.MODAL,
                                   Gtk.MessageType.ERROR,
                                   Gtk.ButtonsType.OK,
                                   _("Couldn't find any %(browser_name)s profiles." % locals()))
        dialog.run()
        dialog.destroy()
