# -*- Mode: Python; coding: utf-8; indent-tabs-mode: nil; tab-width: 4 -*-
### BEGIN LICENSE
# Copyright (C) 2012 Pete Burgers <deltify81@gmail.com>
# This program is free software: you can redistribute it and/or modify it 
# under the terms of the GNU General Public License version 3, as published 
# by the Free Software Foundation.
# 
# This program is distributed in the hope that it will be useful, but 
# WITHOUT ANY WARRANTY; without even the implied warranties of 
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
# PURPOSE.  See the GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License along 
# with this program.  If not, see <http://www.gnu.org/licenses/>.
### END LICENSE

import os
import shutil
import tempfile
import codecs
import re
import logging
logger = logging.getLogger(__name__)

import deltify.settings as settings

SCAN_FILE_ENCODING = "utf-8"

def create_temporary_copy(path, temp_filename):
    temp_dir = tempfile.gettempdir()
    temp_path = os.path.join(temp_dir, temp_filename)
    shutil.copy2(path, temp_path)
    return temp_path

def save_scan(filename, data):
    """ Saves the page contents to the specified file """
    filename = os.path.join(settings.SETTINGS_PATH, filename)
    with codecs.open(filename, "w", SCAN_FILE_ENCODING) as scan_file:
        scan_file.write(data)

def load_scan(filename):
    """ Loads page contents from the specified file """
    filename = os.path.join(settings.SETTINGS_PATH, filename)
    try:
        with codecs.open(filename, "r", SCAN_FILE_ENCODING) as scan_file:
            return scan_file.read()
    except IOError:
        return ""

def move_and_overwrite_scan(source, dest):
    """ Deletes dest and moves source to dest """
    source = os.path.join(settings.SETTINGS_PATH, source)
    dest = os.path.join(settings.SETTINGS_PATH, dest)
    try:
        os.remove(dest)
    except OSError:
        pass
    shutil.move(source, dest)

def add_header(data, insert_style, delete_style):
    """ Set the CSS styles for insert/delete highlighting """
    # baseURI, content_type and charset are handled when the data is loaded into WebKit
#    header = "\n<base href='%s' target='_parent'>\n" % url
             # "<meta http-equiv='Content-Type' content='text/html;"+
             # "charset=UTF-8'>\n" +
    header = "<style type='text/css'>\n span.insdiff {%s}\n del.diff {%s}\n</style>\n" % (insert_style, delete_style)
    match = re.search('<\s*head[^>]*>', data, flags=re.IGNORECASE)
    if match:
        return data[:match.end()] + header + data[match.end():]
    else:
        # No head element, just tack one on the start
        logger.debug("No <head> element found")
        return "<head>%s</head>\n%s" % (header, data)
